<?php
// AVASMS_BUILD_ID: 2025-12-31T09:00Z
namespace Avayemarketing\AvaSMS\Admin;

use Avayemarketing\AvaSMS\Settings\Options;
use Avayemarketing\AvaSMS\Logging\Logger;
use Avayemarketing\AvaSMS\Campaigns\CampaignsModule;

if (!defined('ABSPATH')) exit;

final class AdminModule
{
    private const CAP = 'manage_options';
    private const MENU_SLUG = 'avasms';

    public function register_hooks(): void
    {
        add_action('admin_menu', [$this, 'menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);

        // AJAX
        add_action('wp_ajax_avasms_test_sms', [$this, 'ajax_test_sms']);
        add_action('wp_ajax_avasms_send_single', [$this, 'ajax_send_single']);
        add_action('wp_ajax_avasms_admin_load', [$this, 'ajax_admin_load']);
        add_action('wp_ajax_avasms_send_single_sms', [$this, 'ajax_send_single']);
        add_action('wp_ajax_avasms_clear_logs', [$this, 'ajax_clear_logs']);
    }

    public function menu(): void
    {
        add_menu_page('AvaSMS', 'AvaSMS', self::CAP, self::MENU_SLUG, [$this, 'page_dashboard'], 'dashicons-email-alt2', 58);

        add_submenu_page(self::MENU_SLUG, 'داشبورد', 'داشبورد', self::CAP, self::MENU_SLUG, [$this, 'page_dashboard']);
        add_submenu_page(self::MENU_SLUG, 'تنظیمات درگاه پیامکی', 'تنظیمات درگاه پیامکی', self::CAP, 'avasms-gateway', [$this, 'page_gateway']);
        add_submenu_page(self::MENU_SLUG, 'تنظیمات OTP', 'تنظیمات OTP', self::CAP, 'avasms-otp', [$this, 'page_otp']);
        add_submenu_page(self::MENU_SLUG, 'پیامک‌های ووکامرس', 'پیامک‌های ووکامرس', self::CAP, 'avasms-woo', [$this, 'page_woo']);
        add_submenu_page(self::MENU_SLUG, 'پیام‌های خودکار', 'پیام‌های خودکار', self::CAP, 'avasms-auto', [$this, 'page_auto']);
        add_submenu_page(self::MENU_SLUG, 'ارسال پیام تکی', 'ارسال پیام تکی', self::CAP, 'avasms-single', [$this, 'page_single']);
        add_submenu_page(self::MENU_SLUG, 'کمپین‌ها (ساخت کمپین)', 'کمپین‌ها (ساخت کمپین)', self::CAP, 'avasms-campaign-builder', [$this, 'page_campaign_builder']);
        add_submenu_page(self::MENU_SLUG, 'لیست کمپین‌ها', 'لیست کمپین‌ها', self::CAP, 'avasms-campaigns', [$this, 'page_campaigns_list']);
        add_submenu_page(self::MENU_SLUG, 'لاگ‌ها و گزارشات', 'لاگ‌ها و گزارشات', self::CAP, 'avasms-logs', [$this, 'page_logs']);
    }

    public function register_settings(): void
    {
        register_setting('avasms_gateways_group', Options::OPT_GATEWAYS, [$this, 'sanitize_gateways']);
        register_setting('avasms_otp_group', Options::OPT_OTP, [$this, 'sanitize_otp']);
        register_setting('avasms_woo_group', Options::OPT_WOO, [$this, 'sanitize_woo']);
        register_setting('avasms_campaigns_group', Options::OPT_CAMPAIGNS, [$this, 'sanitize_campaigns']);
    }

    public function enqueue_assets(string $hook): void
    {
        if (strpos($hook, 'avasms') === false) return;
        wp_enqueue_style('avasms-admin', AVASMS_URL . 'assets/admin/admin.css', [], AVASMS_VERSION);
        wp_enqueue_script('avasms-admin', AVASMS_URL . 'assets/admin/admin.js', [], AVASMS_VERSION, true);
    
        wp_localize_script('avasms-admin', 'AvaSMSAdmin', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('avasms_admin'),
          ]);
}

    
    // ---------------- AJAX: load pages without reload (internal navigation) ----------------
    public function ajax_admin_load(): void
    {
        if (!current_user_can(self::CAP)) {
            wp_send_json_error(['message' => 'دسترسی غیرمجاز.'], 403);
        }
        check_ajax_referer('avasms_admin', 'nonce');

        $slug = isset($_POST['page']) ? sanitize_key((string)$_POST['page']) : self::MENU_SLUG;

        $title = '';
        ob_start();
        switch ($slug) {
            case self::MENU_SLUG:
                $title = 'داشبورد AvaSMS';
                $this->render_dashboard_inner();
                break;
            case 'avasms-gateway':
                $title = 'تنظیمات درگاه پیامکی';
                $this->render_gateway_inner();
                break;
            case 'avasms-otp':
                $title = 'تنظیمات OTP';
                $this->render_otp_inner();
                break;
            case 'avasms-woo':
                $title = 'پیامک‌های ووکامرس';
                $this->render_woo_inner();
                break;
            case 'avasms-auto':
                $title = 'پیام‌های خودکار';
                $this->render_auto_inner();
                break;
            case 'avasms-campaigns':
                $title = 'لیست کمپین‌ها';
                $this->render_campaigns_list_inner();
                break;
            case 'avasms-logs':
                $title = 'لاگ‌ها و گزارشات';
                $this->render_logs_inner();
                break;
            default:
                $title = 'AvaSMS';
                echo '<div class="notice notice-error"><p>صفحه نامعتبر است.</p></div>';
                break;
        }
        $html = ob_get_clean();

        wp_send_json_success([
            'title' => $title,
            'slug'  => $slug,
            'html'  => $html,
        ]);
    }

// ---------------- Sanitizers ----------------
    public function sanitize_gateways($input): array
    {
        $in = is_array($input) ? $input : [];
        $def = Options::gateways();

        // accept both active and active_gateway
        $active = '';
        if (isset($in['active_gateway'])) $active = sanitize_key((string)$in['active_gateway']);
        elseif (isset($in['active'])) $active = sanitize_key((string)$in['active']);
        else $active = (string)($def['active_gateway'] ?? 'kavenegar');

        if ($active === 'melli_sms') $active = 'melli_sms';
        if (!in_array($active, ['kavenegar','ippanel','farazsms','melli_sms'], true)) $active = 'kavenegar';

        $out = $def;
        $out['active_gateway'] = $active;

        // Kavenegar
        if (isset($in['kavenegar']) && is_array($in['kavenegar'])) {
            $g = $in['kavenegar'];
            $out['kavenegar']['use_pattern']   = !empty($g['use_pattern']) ? 1 : 0;
            $out['kavenegar']['api_key']       = sanitize_text_field((string)($g['api_key'] ?? ''));
            $out['kavenegar']['sender_line']   = sanitize_text_field((string)($g['sender_line'] ?? ''));
            $out['kavenegar']['template_code'] = sanitize_text_field((string)($g['template_code'] ?? ''));
            $out['kavenegar']['sample_text']   = sanitize_textarea_field((string)($g['sample_text'] ?? ''));
        }

        // IPPANEL
        if (isset($in['ippanel']) && is_array($in['ippanel'])) {
            $g = $in['ippanel'];
            $out['ippanel']['use_pattern']   = !empty($g['use_pattern']) ? 1 : 0;
            $out['ippanel']['api_key']       = sanitize_text_field((string)($g['api_key'] ?? ''));
            $out['ippanel']['sender_line']   = sanitize_text_field((string)($g['sender_line'] ?? ''));
            $out['ippanel']['template_code'] = sanitize_text_field((string)($g['template_code'] ?? ''));
            $out['ippanel']['endpoint']      = esc_url_raw((string)($g['endpoint'] ?? $out['ippanel']['endpoint']));
            $out['ippanel']['sample_text']   = sanitize_textarea_field((string)($g['sample_text'] ?? ''));
        }

        // FarazSMS
        if (isset($in['farazsms']) && is_array($in['farazsms'])) {
            $g = $in['farazsms'];
            $out['farazsms']['use_pattern']   = !empty($g['use_pattern']) ? 1 : 0;
            $out['farazsms']['username']      = sanitize_text_field((string)($g['username'] ?? ''));
            $out['farazsms']['password']      = sanitize_text_field((string)($g['password'] ?? ''));
            $out['farazsms']['sender_line']   = sanitize_text_field((string)($g['sender_line'] ?? ''));
            $out['farazsms']['template_code'] = sanitize_text_field((string)($g['template_code'] ?? ''));
            $out['farazsms']['endpoint']      = esc_url_raw((string)($g['endpoint'] ?? ''));
            $out['farazsms']['sample_text']   = sanitize_textarea_field((string)($g['sample_text'] ?? ''));
        }

        // MelliSMS
        if (isset($in['melli_sms']) && is_array($in['melli_sms'])) {
            $g = $in['melli_sms'];
            $out['melli_sms']['use_pattern']   = !empty($g['use_pattern']) ? 1 : 0;
            $out['melli_sms']['username']      = sanitize_text_field((string)($g['username'] ?? ''));
            $out['melli_sms']['password']      = sanitize_text_field((string)($g['password'] ?? ''));
            $out['melli_sms']['sender_line']   = sanitize_text_field((string)($g['sender_line'] ?? ''));
            $out['melli_sms']['template_code'] = sanitize_text_field((string)($g['template_code'] ?? ''));
            $out['melli_sms']['endpoint']      = esc_url_raw((string)($g['endpoint'] ?? ''));
            $out['melli_sms']['sample_text']   = sanitize_textarea_field((string)($g['sample_text'] ?? ''));
        }

        // support legacy key melli_sms -> melli_sms
        if (isset($in['melli_sms']) && is_array($in['melli_sms']) && empty($in['melli_sms'])) {
            $g = $in['melli_sms'];
            $out['melli_sms']['use_pattern']   = !empty($g['use_pattern']) ? 1 : 0;
            $out['melli_sms']['username']      = sanitize_text_field((string)($g['username'] ?? ''));
            $out['melli_sms']['password']      = sanitize_text_field((string)($g['password'] ?? ''));
            $out['melli_sms']['sender_line']   = sanitize_text_field((string)($g['sender_line'] ?? ''));
            $out['melli_sms']['template_code'] = sanitize_text_field((string)($g['template_code'] ?? ''));
            $out['melli_sms']['endpoint']      = esc_url_raw((string)($g['endpoint'] ?? ''));
            $out['melli_sms']['sample_text']   = sanitize_textarea_field((string)($g['sample_text'] ?? ''));
        }

        return $out;
    }

    public function sanitize_otp($input): array
    {
        $in = is_array($input) ? $input : [];
        $def = Options::otp();

        $enabled = !empty($in['enabled']) ? 1 : 0;
        $length  = max(4, min(8, (int)($in['length'] ?? $def['length'])));
        $expiry  = max(30, min(900, (int)($in['expiry_seconds'] ?? $def['expiry_seconds'])));
        $max_send = max(1, min(30, (int)($in['max_send_per_hour'] ?? $def['max_send_per_hour'])));
        $max_try = max(3, min(50, (int)($in['max_verify_attempts'] ?? $def['max_verify_attempts'])));
        $lock = max(1, min(1440, (int)($in['lock_minutes'] ?? $def['lock_minutes'])));
        $tpl = sanitize_textarea_field((string)($in['message_template'] ?? $def['message_template'])) ?: (string)$def['message_template'];

        $allow_email = !empty($in['allow_login_with_email']) ? 1 : 0;

        $turnstile_enabled = !empty($in['turnstile_enabled']) ? 1 : 0;
        $turnstile_site_key = sanitize_text_field((string)($in['turnstile_site_key'] ?? $def['turnstile_site_key'] ?? ''));
        $turnstile_secret_key = sanitize_text_field((string)($in['turnstile_secret_key'] ?? $def['turnstile_secret_key'] ?? ''));

        return [
            'enabled' => $enabled,
            'length' => $length,
            'expiry_seconds' => $expiry,
            'max_send_per_hour' => $max_send,
            'max_verify_attempts' => $max_try,
            'lock_minutes' => $lock,
            'message_template' => $tpl,

            'allow_login_with_email' => $allow_email,

            'turnstile_enabled' => $turnstile_enabled,
            'turnstile_site_key' => $turnstile_site_key,
            'turnstile_secret_key' => $turnstile_secret_key,
        ];
    }

    public function sanitize_woo($input): array
    {
        $in = is_array($input) ? $input : [];
        $def = Options::woo();
        $out = $def;

        $out['enabled'] = !empty($in['enabled']) ? 1 : 0;

        $out['customer_enabled'] = !empty($in['customer_enabled']) ? 1 : 0;
        $out['customer_on_new_order'] = !empty($in['customer_on_new_order']) ? 1 : 0;
        $out['customer_on_status_change'] = !empty($in['customer_on_status_change']) ? 1 : 0;

        $statuses = ['processing','completed','on-hold','cancelled','refunded','failed'];
        $out['customer_statuses'] = [];
        foreach ($statuses as $st) {
            $key = str_replace('-', '_', $st);
            $out['customer_statuses'][$st] = !empty(($in['customer_statuses'] ?? [])[$key]) ? 1 : 0;
        }

        $out['admin_enabled'] = !empty($in['admin_enabled']) ? 1 : 0;
        $out['admin_on_new_order'] = !empty($in['admin_on_new_order']) ? 1 : 0;
        $out['admin_on_status_change'] = !empty($in['admin_on_status_change']) ? 1 : 0;
        $out['admin_phones'] = sanitize_text_field((string)($in['admin_phones'] ?? ''));

        $out['admin_statuses'] = [];
        foreach ($statuses as $st) {
            $key = str_replace('-', '_', $st);
            $out['admin_statuses'][$st] = !empty(($in['admin_statuses'] ?? [])[$key]) ? 1 : 0;
        }

        $out['tpl_customer_new_order'] = sanitize_textarea_field((string)($in['tpl_customer_new_order'] ?? $def['tpl_customer_new_order']));
        $out['tpl_customer_status']    = sanitize_textarea_field((string)($in['tpl_customer_status'] ?? $def['tpl_customer_status']));
        $out['tpl_admin_new_order']    = sanitize_textarea_field((string)($in['tpl_admin_new_order'] ?? $def['tpl_admin_new_order']));
        $out['tpl_admin_status']       = sanitize_textarea_field((string)($in['tpl_admin_status'] ?? $def['tpl_admin_status']));

        return $out;
    }

    public function sanitize_campaigns($input): array
    {
        $in = is_array($input) ? $input : [];
        $def = Options::campaigns();

        return [
            'batch_size'            => max(10, min(200, (int)($in['batch_size'] ?? $def['batch_size']))),
            'require_optin'         => !empty($in['require_optin']) ? 1 : 0,

            'enable_birthday'       => !empty($in['enable_birthday']) ? 1 : 0,
            'birthday_message'      => sanitize_textarea_field((string)($in['birthday_message'] ?? $def['birthday_message'])),
            'birthday_coupon'       => sanitize_text_field((string)($in['birthday_coupon'] ?? '')),

            'birthday_auto_coupon'  => !empty($in['birthday_auto_coupon']) ? 1 : 0,
            'birthday_discount_pct' => max(1, min(90, (int)($in['birthday_discount_pct'] ?? $def['birthday_discount_pct']))),
            'birthday_expiry_days'  => max(1, min(365, (int)($in['birthday_expiry_days'] ?? $def['birthday_expiry_days']))),
            'birthday_usage_limit'  => max(1, min(10, (int)($in['birthday_usage_limit'] ?? $def['birthday_usage_limit']))),
            'birthday_min_amount'   => max(0, (float)($in['birthday_min_amount'] ?? $def['birthday_min_amount'])),
            'birthday_free_shipping'=> !empty($in['birthday_free_shipping']) ? 1 : 0,
        ];
    }

    // ---------------- UI Shell ----------------
    private function render_logs_inner(): void
    {
        $nonce = wp_create_nonce('avasms_logs');
            $level = isset($_GET['level']) ? sanitize_key((string)$_GET['level']) : '';
            if (!in_array($level, ['info','warning','error'], true)) $level = '';

            echo '<div class="avasms-panel">';
            echo '<h2>لاگ‌ها</h2>';

            echo '<div style="display:flex; gap:10px; align-items:center; flex-wrap:wrap; margin:10px 0;">';
            echo '<form method="get" style="display:flex; gap:8px; align-items:center;">';
            echo '<input type="hidden" name="page" value="avasms-logs">';
            echo '<select name="level">';
            echo '<option value="">همه</option>';
            echo '<option value="info" ' . selected('info', $level, false) . '>موفق</option>';
            echo '<option value="warning" ' . selected('warning', $level, false) . '>هشدار</option>';
            echo '<option value="error" ' . selected('error', $level, false) . '>ناموفق</option>';
            echo '</select>';
            echo '<button class="button">فیلتر</button>';
            echo '</form>';

            echo '<button type="button" class="button" id="avasms_clear_logs" data-nonce="' . esc_attr($nonce) . '">پاکسازی لاگ‌های قدیمی (۳۰ روز)</button>';
            echo '<span id="avasms_clear_logs_result" class="avasms-status"></span>';
            echo '</div>';

            $rows = Logger::recent(300, $level ?: null);

            echo '<table class="widefat striped"><thead><tr>';
            echo '<th>زمان</th><th>سطح</th><th>بخش</th><th>پیام</th><th>داده</th>';
            echo '</tr></thead><tbody>';

            if (empty($rows)) {
                echo '<tr><td colspan="5">لاگی ثبت نشده.</td></tr>';
            } else {
                foreach ($rows as $r) {
                    echo '<tr>';
                    echo '<td>' . esc_html((string)$r['created_at']) . '</td>';
                    echo '<td><code>' . esc_html((string)$r['level']) . '</code></td>';
                    echo '<td>' . esc_html((string)$r['context']) . '</td>';
                    echo '<td>' . esc_html((string)$r['message']) . '</td>';
                    echo '<td><code style="white-space:pre-wrap; display:block; max-width:520px;">' . esc_html((string)$r['data']) . '</code></td>';
                    echo '</tr>';
                }
            }

            echo '</tbody></table>';
            echo '</div>';
    }

    private function render_campaigns_list_inner(): void
    {
$nonce = wp_create_nonce('avasms_campaign');
            $mod = new CampaignsModule();
            $rows = $mod->list_campaigns(200);

            echo '<div class="avasms-panel">';
            echo '<h2>لیست کمپین‌ها</h2>';
            echo '<input type="hidden" id="avasms_campaign_nonce" value="' . esc_attr($nonce) . '">';

            echo '<table class="widefat striped"><thead><tr>';
            echo '<th>ID</th><th>عنوان</th><th>وضعیت</th><th>زمان‌بندی</th><th>ایجاد</th><th>عملیات</th>';
            echo '</tr></thead><tbody>';

            if (empty($rows)) {
                echo '<tr><td colspan="6">کمپینی ثبت نشده.</td></tr>';
            } else {
                foreach ($rows as $r) {
                    $id = (int)$r['id'];
                    echo '<tr>';
                    echo '<td>' . esc_html((string)$id) . '</td>';
                    echo '<td>' . esc_html((string)$r['title']) . '</td>';
                    echo '<td><code>' . esc_html((string)$r['status']) . '</code></td>';
                    echo '<td>' . esc_html((string)($r['scheduled_at'] ?? '—')) . '</td>';
                    echo '<td>' . esc_html((string)($r['created_at'] ?? '—')) . '</td>';
                    echo '<td>';
                    echo '<button class="button avasms-c-prepare" data-id="' . esc_attr((string)$id) . '">آماده‌سازی لیست</button> ';
                    echo '<button class="button button-primary avasms-c-send" data-id="' . esc_attr((string)$id) . '">ارسال (Batch)</button> ';
                    echo '<button class="button avasms-c-delete" data-id="' . esc_attr((string)$id) . '">حذف</button>';
                    echo '<div class="avasms-c-progress" id="avasms_c_prog_' . esc_attr((string)$id) . '" style="margin-top:6px;"></div>';
                    echo '</td>';
                    echo '</tr>';
                }
            }

            echo '</tbody></table>';
            echo '<p class="description">ارسال Batch هر بار به تعداد تنظیم‌شده (پیش‌فرض 20) ارسال می‌کند. برای ارسال کامل، دکمه را چند بار بزنید تا Pending صفر شود یا از زمان‌بندی استفاده کنید.</p>';
            echo '</div>';
    }

    private function render_woo_inner(): void
    {
$opt = Options::woo();
            echo '<div class="avasms-panel"><h2>تنظیمات پیشرفته پیامک ووکامرس</h2><p class="avasms-help">فعال/غیرفعال‌سازی دقیق برای مشتری و مدیر، بر اساس رویداد و وضعیت سفارش.</p>';
            echo '<form method="post" action="options.php">';
            settings_fields('avasms_woo_group');
            echo '<div class="avasms-fields">';
            $this->checkbox_row('فعال‌سازی کلی پیامک ووکامرس', Options::OPT_WOO.'[enabled]', (int)$opt['enabled']);
            echo '</div><div class="avasms-divider"></div>';

            echo '<h2 style="margin:0 0 10px;">پیامک مشتری</h2>';
            echo '<div class="avasms-fields">';
            $this->checkbox_row('فعال‌سازی پیامک مشتری', Options::OPT_WOO.'[customer_enabled]', (int)$opt['customer_enabled']);
            $this->checkbox_row('ثبت سفارش (مشتری)', Options::OPT_WOO.'[customer_on_new_order]', (int)$opt['customer_on_new_order']);
            $this->checkbox_row('تغییر وضعیت سفارش (مشتری)', Options::OPT_WOO.'[customer_on_status_change]', (int)$opt['customer_on_status_change']);
            echo '<label>وضعیت‌های فعال</label><div class="avasms-inline">';
            $map = ['processing'=>'در حال انجام','completed'=>'تکمیل شده','on-hold'=>'در انتظار','cancelled'=>'لغو شده','refunded'=>'مرجوع شده','failed'=>'ناموفق'];
            foreach ($map as $k=>$lab) {
                $key = str_replace('-', '_', $k);
                $val = (int)($opt['customer_statuses'][$k] ?? 0);
                echo '<label style="display:inline-flex;gap:6px;align-items:center;margin-left:10px;">';
                echo '<input type="checkbox" name="'.esc_attr(Options::OPT_WOO).'[customer_statuses]['.esc_attr($key).']" value="1" '.checked(1,$val,false).'>';
                echo '<span>'.esc_html($lab).'</span></label>';
            }
            echo '</div>';
            $this->field_row('متن پیام ثبت سفارش مشتری', Options::OPT_WOO.'[tpl_customer_new_order]', (string)$opt['tpl_customer_new_order'], 'textarea', 'متغیرها: {order_id} {price} {status}');
            $this->field_row('متن پیام تغییر وضعیت مشتری', Options::OPT_WOO.'[tpl_customer_status]', (string)$opt['tpl_customer_status'], 'textarea', 'متغیرها: {order_id} {price} {status}');
            echo '</div><div class="avasms-divider"></div>';

            echo '<h2 style="margin:0 0 10px;">پیامک مدیر</h2>';
            echo '<div class="avasms-fields">';
            $this->checkbox_row('فعال‌سازی پیامک مدیر', Options::OPT_WOO.'[admin_enabled]', (int)$opt['admin_enabled']);
            $this->field_row('شماره(های) مدیر', Options::OPT_WOO.'[admin_phones]', (string)$opt['admin_phones'], 'text', 'با کاما جدا کنید: 0912...,0935...');
            $this->checkbox_row('ثبت سفارش (مدیر)', Options::OPT_WOO.'[admin_on_new_order]', (int)$opt['admin_on_new_order']);
            $this->checkbox_row('تغییر وضعیت سفارش (مدیر)', Options::OPT_WOO.'[admin_on_status_change]', (int)$opt['admin_on_status_change']);
            echo '<label>وضعیت‌های فعال</label><div class="avasms-inline">';
            foreach ($map as $k=>$lab) {
                $key = str_replace('-', '_', $k);
                $val = (int)($opt['admin_statuses'][$k] ?? 0);
                echo '<label style="display:inline-flex;gap:6px;align-items:center;margin-left:10px;">';
                echo '<input type="checkbox" name="'.esc_attr(Options::OPT_WOO).'[admin_statuses]['.esc_attr($key).']" value="1" '.checked(1,$val,false).'>';
                echo '<span>'.esc_html($lab).'</span></label>';
            }
            echo '</div>';
            $this->field_row('متن پیام ثبت سفارش مدیر', Options::OPT_WOO.'[tpl_admin_new_order]', (string)$opt['tpl_admin_new_order'], 'textarea', 'متغیرها: {order_id} {price} {status}');
            $this->field_row('متن پیام تغییر وضعیت مدیر', Options::OPT_WOO.'[tpl_admin_status]', (string)$opt['tpl_admin_status'], 'textarea', 'متغیرها: {order_id} {price} {status}');
            echo '</div><div class="avasms-divider"></div>';

            submit_button('ذخیره تنظیمات', 'primary');
            echo '</form></div>';
    }

    private function render_auto_inner(): void
    {
$opt = Options::campaigns();
            echo '<div class="avasms-panel"><h2>تبریک تولد + کد تخفیف</h2><p class="avasms-help">ارسال تبریک تولد براساس تاریخ تولد ثبت‌شده در حساب کاربری.</p>';
            echo '<form method="post" action="options.php">';
            settings_fields('avasms_campaigns_group');
            echo '<div class="avasms-fields">';
            $this->checkbox_row('فعال‌سازی تبریک تولد', Options::OPT_CAMPAIGNS.'[enable_birthday]', (int)$opt['enable_birthday']);
            $this->checkbox_row('تولید خودکار کوپن', Options::OPT_CAMPAIGNS.'[birthday_auto_coupon]', (int)$opt['birthday_auto_coupon']);
            $this->field_row('درصد تخفیف', Options::OPT_CAMPAIGNS.'[birthday_discount_pct]', (string)$opt['birthday_discount_pct'], 'number');
            $this->field_row('اعتبار کوپن (روز)', Options::OPT_CAMPAIGNS.'[birthday_expiry_days]', (string)$opt['birthday_expiry_days'], 'number');
            $this->field_row('حداکثر استفاده', Options::OPT_CAMPAIGNS.'[birthday_usage_limit]', (string)$opt['birthday_usage_limit'], 'number');
            $this->field_row('حداقل مبلغ سفارش', Options::OPT_CAMPAIGNS.'[birthday_min_amount]', (string)$opt['birthday_min_amount'], 'number');
            $this->checkbox_row('ارسال رایگان', Options::OPT_CAMPAIGNS.'[birthday_free_shipping]', (int)$opt['birthday_free_shipping']);
            $this->field_row('کد کوپن ثابت (اختیاری)', Options::OPT_CAMPAIGNS.'[birthday_coupon]', (string)$opt['birthday_coupon'], 'text', 'اگر تولید خودکار خاموش باشد استفاده می‌شود');
            $this->field_row('متن پیام تبریک تولد', Options::OPT_CAMPAIGNS.'[birthday_message]', (string)$opt['birthday_message'], 'textarea', 'متغیرها: {first_name} {coupon} {discount} {expires}');
            echo '</div><div class="avasms-divider"></div>';
            submit_button('ذخیره تنظیمات', 'primary');
            echo '</form></div>';

            echo '<div class="avasms-panel"><h2>رضایت پیامکی (Opt-in)</h2><p class="avasms-help">برای ارسال تبلیغاتی/کمپین می‌توانید الزام رضایت را فعال کنید.</p>';
            echo '<form method="post" action="options.php">';
            settings_fields('avasms_campaigns_group');
            echo '<div class="avasms-fields">';
            $this->checkbox_row('الزام رضایت برای پیامک‌های تبلیغاتی', Options::OPT_CAMPAIGNS.'[require_optin]', (int)$opt['require_optin']);
            $this->field_row('Batch Size کمپین', Options::OPT_CAMPAIGNS.'[batch_size]', (string)$opt['batch_size'], 'number');
            echo '</div><div class="avasms-divider"></div>';
            submit_button('ذخیره', 'secondary');
            echo '</form></div>';
    }

    private function render_otp_inner(): void
    {
$opt = Options::otp();
            echo '<div class="avasms-panel"><h2>تنظیمات کلی OTP</h2><p class="avasms-help">برای جلوگیری از brute force و سوءاستفاده، محدودیت‌ها را تنظیم کنید.</p>';
            echo '<form method="post" action="options.php">';
            settings_fields('avasms_otp_group');
            echo '<div class="avasms-fields">';
            $this->checkbox_row('فعال‌سازی OTP', Options::OPT_OTP.'[enabled]', (int)$opt['enabled']);
            $this->field_row('طول کد', Options::OPT_OTP.'[length]', (string)$opt['length'], 'number', 'بین ۴ تا ۸');
            $this->field_row('اعتبار کد (ثانیه)', Options::OPT_OTP.'[expiry_seconds]', (string)$opt['expiry_seconds'], 'number', 'مثلاً ۱۲۰');
            $this->field_row('حداکثر ارسال در ساعت (هر شماره)', Options::OPT_OTP.'[max_send_per_hour]', (string)$opt['max_send_per_hour'], 'number', 'Rate limit');
            $this->field_row('حداکثر تلاش ورود', Options::OPT_OTP.'[max_verify_attempts]', (string)$opt['max_verify_attempts'], 'number', 'پس از این تعداد، قفل می‌شود');
            $this->field_row('مدت قفل (دقیقه)', Options::OPT_OTP.'[lock_minutes]', (string)$opt['lock_minutes'], 'number', 'مثلاً ۱۵');
            $this->field_row('قالب پیام OTP', Options::OPT_OTP.'[message_template]', (string)$opt['message_template'], 'textarea', 'متغیرها: {code} و {minutes}');

            echo '<div class="avasms-divider"></div>';
            echo '<h2 style="margin:0 0 10px;">ورود با موبایل یا ایمیل</h2>';
            $this->checkbox_row('اجازه ورود با ایمیل (در کنار موبایل)', Options::OPT_OTP.'[allow_login_with_email]', (int)($opt['allow_login_with_email'] ?? 1), '');

            echo '<div class="avasms-divider"></div>';
            echo '<h2 style="margin:0 0 10px;">کپچا Cloudflare Turnstile</h2>';
            $this->checkbox_row('فعال‌سازی Turnstile', Options::OPT_OTP.'[turnstile_enabled]', (int)($opt['turnstile_enabled'] ?? 0));
            $this->field_row('Site Key', Options::OPT_OTP.'[turnstile_site_key]', (string)($opt['turnstile_site_key'] ?? ''), 'text', 'کلید عمومی Turnstile');
            $this->field_row('Secret Key', Options::OPT_OTP.'[turnstile_secret_key]', (string)($opt['turnstile_secret_key'] ?? ''), 'text', 'کلید خصوصی Turnstile');
            echo '</div><div class="avasms-divider"></div>';
            submit_button('ذخیره تنظیمات', 'primary');
            echo '</form></div>';
    }

    private function render_gateway_inner(): void
    {
$opt = Options::gateways();
            $active = (string)($opt['active'] ?? 'kavenegar');

            echo '<div class="avasms-panel"><h2>انتخاب و تنظیم درگاه</h2><p class="avasms-help">درگاه را انتخاب کنید؛ فقط تنظیمات همان درگاه نمایش داده می‌شود.</p>';
            echo '<form method="post" action="options.php">';
            settings_fields('avasms_gateways_group');

            echo '<div class="avasms-fields">';
            echo '<label for="avasms_gateway_active">درگاه فعال</label>';
            echo '<select id="avasms_gateway_active" name="'.esc_attr(Options::OPT_GATEWAYS).'[active]">';
            foreach (['kavenegar'=>'کاونگار','ippanel'=>'IPPanel','farazsms'=>'فراز اس‌ام‌اس','melli_sms'=>'ملی پیامک'] as $k=>$lab) {
                echo '<option value="'.esc_attr($k).'" '.selected($active,$k,false).'>'.esc_html($lab).'</option>';
            }
            echo '</select></div><div class="avasms-divider"></div>';

            echo $this->gateway_panel_kavenegar($opt);
            echo $this->gateway_panel_ippanel($opt);
            echo $this->gateway_panel_farazsms($opt);
            echo $this->gateway_panel_melli_sms($opt);

            echo '<div class="avasms-divider"></div>';
            submit_button('ذخیره تنظیمات','primary');
            echo '</form></div>';

            echo '<div class="avasms-panel"><h2>تست تنظیمات درگاه</h2><p class="avasms-help">شماره موبایل را وارد کنید و پیام تست ارسال کنید. نتیجه در لاگ‌ها هم ذخیره می‌شود.</p>';
            echo '<div class="avasms-inline">';
            echo '<input id="avasms_test_phone" type="text" placeholder="09120000000" class="regular-text">';
            echo '<button class="button button-secondary" id="avasms_test_btn" data-nonce="'.esc_attr(wp_create_nonce('avasms_test_sms')).'">ارسال پیام تست</button>';
            echo '<span class="avasms-status"><span class="avasms-dot"></span><span id="avasms_test_result">—</span></span>';
            echo '</div></div>';
    }

    private function render_dashboard_inner(): void
    {
global $wpdb;
            $queue = CampaignsModule::table_queue();
            $sent = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$queue} WHERE status='sent'");
            $failed = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$queue} WHERE status='failed'");
            $pending = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$queue} WHERE status='pending'");
            $log_today = (int) $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM " . Logger::table_name() . " WHERE created_at >= %s", date('Y-m-d 00:00:00', current_time('timestamp'))));

            echo '<div class="avasms-cardgrid">';
            $this->stat_card('پیامک‌های موفق', number_format_i18n($sent), 'dashicons-yes');
            $this->stat_card('پیامک‌های ناموفق', number_format_i18n($failed), 'dashicons-warning');
            $this->stat_card('در صف ارسال', number_format_i18n($pending), 'dashicons-update');
            $this->stat_card('لاگ امروز', number_format_i18n($log_today), 'dashicons-admin-site');
            echo '</div>';

            echo '<div class="avasms-panel"><h2>شروع سریع</h2><p class="avasms-help">برای کارکرد صحیح ابتدا درگاه پیامکی را تنظیم کنید، سپس پیامک‌های ووکامرس و پیام‌های خودکار را فعال کنید.</p>';
            echo '<div class="avasms-inline">';
            echo '<a class="button button-primary" href="'.esc_url(admin_url('admin.php?page=avasms-gateway')).'">درگاه پیامکی</a>';
            echo '<a class="button button-secondary" href="'.esc_url(admin_url('admin.php?page=avasms-woo')).'">پیامک ووکامرس</a>';
            echo '<a class="button button-secondary" href="'.esc_url(admin_url('admin.php?page=avasms-auto')).'">پیام‌های خودکار</a>';
            echo '</div></div>';

            $rows = Logger::recent(15);
            echo '<div class="avasms-panel"><h2>لاگ‌های اخیر</h2><table class="widefat striped"><thead><tr><th>زمان</th><th>سطح</th><th>بخش</th><th>پیام</th></tr></thead><tbody>';
            if (empty($rows)) echo '<tr><td colspan="4">لاگی ثبت نشده.</td></tr>';
            else foreach ($rows as $r) {
                echo '<tr><td>'.esc_html((string)$r['created_at']).'</td><td>'.esc_html((string)$r['level']).'</td><td>'.esc_html((string)$r['context']).'</td><td>'.esc_html((string)$r['message']).'</td></tr>';
            }
            echo '</tbody></table></div>';
    }

function shell(string $active_slug, string $title, callable $content): void
    {
        $nav = [
            self::MENU_SLUG      => ['داشبورد', 'dashicons-dashboard'],
            'avasms-gateway'     => ['درگاه پیامکی', 'dashicons-email'],
            'avasms-otp'         => ['تنظیمات OTP', 'dashicons-shield'],
            'avasms-woo'         => ['پیامک‌های ووکامرس', 'dashicons-cart'],
            'avasms-auto'        => ['پیام‌های خودکار', 'dashicons-controls-repeat'],
            'avasms-campaigns'   => ['کمپین‌ها', 'dashicons-megaphone'],
            'avasms-logs'        => ['لاگ‌ها', 'dashicons-list-view'],
        ];

        echo '<div class="wrap avasms-wrap">';
        echo '<h1 id="avasms-admin-title">' . esc_html($title) . '</h1>';
        echo '<div class="avasms-shell">';
        echo '<div class="avasms-side"><div class="avasms-nav">';
        foreach ($nav as $slug => $meta) {
            $url = admin_url('admin.php?page=' . $slug);
            $cls = ($active_slug === $slug) ? 'is-active' : '';
            echo '<a class="' . esc_attr($cls) . '" href="' . esc_url($url) . '">';
            echo '<span class="dashicons ' . esc_attr($meta[1]) . '"></span>';
            echo '<span>' . esc_html($meta[0]) . '</span>';
            if ($slug === 'avasms-gateway') echo '<span class="avasms-badge">SMS</span>';
            echo '</a>';
        }
        echo '</div></div><div class="avasms-main" id="avasms-admin-main" data-active="'.esc_attr($active_slug).'">';
        $content();
        echo '</div></div></div>';
    }

    private function stat_card(string $title, string $value, string $dashicon): void
    {
        echo '<div class="avasms-card"><div><h3>'.esc_html($title).'</h3><div class="num">'.esc_html($value).'</div></div><div class="icon"><span class="dashicons '.esc_attr($dashicon).'"></span></div></div>';
    }

    private function checkbox_row(string $label, string $name, int $checked, string $hint=''): void
    {
        echo '<label>'.esc_html($label).'</label><div>';
        echo '<label style="display:inline-flex;gap:8px;align-items:center;">';
        echo '<input type="checkbox" name="'.esc_attr($name).'" value="1" '.checked(1,$checked,false).'> <span>فعال</span>';
        echo '</label>';
        if ($hint) echo '<p class="description">'.esc_html($hint).'</p>';
        echo '</div>';
    }

    private function field_row(string $label, string $name, string $value, string $type='text', string $hint=''): void
    {
        echo '<label>'.esc_html($label).'</label><div>';
        if ($type === 'textarea') {
            echo '<textarea rows="3" name="'.esc_attr($name).'">'.esc_textarea($value).'</textarea>';
        } else {
            echo '<input type="'.esc_attr($type).'" name="'.esc_attr($name).'" value="'.esc_attr($value).'">';
        }
        if ($hint) echo '<p class="description">'.esc_html($hint).'</p>';
        echo '</div>';
    }

    // ---------------- Pages ----------------
    public function page_dashboard(): void
    {
        $this->shell(self::MENU_SLUG, 'داشبورد AvaSMS', [$this, 'render_dashboard_inner']);
    }


    public function page_gateway(): void
    {
        $this->shell('avasms-gateway', 'تنظیمات درگاه پیامکی', [$this, 'render_gateway_inner']);
    }


    private function gateway_common_fields(string $gateway_key, array $g): string
    {
        $base = Options::OPT_GATEWAYS . '[' . $gateway_key . ']';
        ob_start();
        echo '<div class="avasms-fields">';
        $this->checkbox_row('ارسال بر اساس الگو (Pattern/Template)', $base.'[use_pattern]', (int)($g['use_pattern'] ?? 0), 'اگر فعال باشد، ارسال از طریق الگو انجام می‌شود؛ در غیر اینصورت ارسال متنی است.');
        $this->field_row('شماره خط (Sender)', $base.'[sender_line]', (string)($g['sender_line'] ?? ''), 'text', 'شماره/خط ارسال‌کننده یا Sender.');
        $this->field_row('کد الگو (Template/Pattern Code)', $base.'[template_code]', (string)($g['template_code'] ?? ''), 'text', 'در صورت استفاده از الگو، کد الگو را وارد کنید.');
        $this->field_row('نمونه متن (Preview)', $base.'[sample_text]', (string)($g['sample_text'] ?? ''), 'textarea', 'نمونه متن برای پیش‌نمایش (ذخیره می‌شود).');
        echo '</div>';
        return (string)ob_get_clean();
    }

    private function gateway_panel_kavenegar(array $opt): string
    {
        $g = $opt['kavenegar'] ?? [];
        $base = Options::OPT_GATEWAYS . '[kavenegar]';
        ob_start();
        echo '<div class="avasms-panel" data-avasms-gateway-panel="kavenegar"><h2>کاونگار</h2><p class="avasms-help">API Key و تنظیمات ارسال را وارد کنید.</p>';
        echo '<div class="avasms-fields">';
        $this->field_row('API Key', $base.'[api_key]', (string)($g['api_key'] ?? ''), 'password');
        echo '</div>';
        echo $this->gateway_common_fields('kavenegar', $g);
        echo '</div>';
        return (string)ob_get_clean();
    }

    private function gateway_panel_ippanel(array $opt): string
    {
        $g = $opt['ippanel'] ?? [];
        $base = Options::OPT_GATEWAYS . '[ippanel]';
        ob_start();
        echo '<div class="avasms-panel" data-avasms-gateway-panel="ippanel"><h2>IPPanel</h2><p class="avasms-help">API Key و تنظیمات ارسال را وارد کنید.</p>';
        echo '<div class="avasms-fields">';
        $this->field_row('API Key', $base.'[api_key]', (string)($g['api_key'] ?? ''), 'password');
        echo '</div>';
        echo $this->gateway_common_fields('ippanel', $g);
        echo '</div>';
        return (string)ob_get_clean();
    }

    private function gateway_panel_farazsms(array $opt): string
    {
        $g = $opt['farazsms'] ?? [];
        $base = Options::OPT_GATEWAYS . '[farazsms]';
        ob_start();
        echo '<div class="avasms-panel" data-avasms-gateway-panel="farazsms"><h2>فراز اس‌ام‌اس</h2><p class="avasms-help">نام کاربری/رمز عبور و تنظیمات ارسال را وارد کنید.</p>';
        echo '<div class="avasms-fields">';
        $this->field_row('نام کاربری', $base.'[username]', (string)($g['username'] ?? ''), 'text');
        $this->field_row('رمز عبور', $base.'[password]', (string)($g['password'] ?? ''), 'password');
        echo '</div>';
        echo $this->gateway_common_fields('farazsms', $g);
        echo '</div>';
        return (string)ob_get_clean();
    }

    private function gateway_panel_melli_sms(array $opt): string
    {
        $g = $opt['melli_sms'] ?? [];
        $base = Options::OPT_GATEWAYS . '[melli_sms]';
        ob_start();
        echo '<div class="avasms-panel" data-avasms-gateway-panel="melli_sms"><h2>ملی پیامک</h2><p class="avasms-help">نام کاربری/رمز عبور و تنظیمات ارسال را وارد کنید.</p>';
        echo '<div class="avasms-fields">';
        $this->field_row('نام کاربری', $base.'[username]', (string)($g['username'] ?? ''), 'text');
        $this->field_row('رمز عبور', $base.'[password]', (string)($g['password'] ?? ''), 'password');
        echo '</div>';
        echo $this->gateway_common_fields('melli_sms', $g);
        echo '</div>';
        return (string)ob_get_clean();
    }

    private function gateway_panel(string $id, string $title, array $fields): string
    {
        ob_start();
        echo '<div class="avasms-panel" data-avasms-gateway-panel="'.esc_attr($id).'"><h2>'.esc_html($title).'</h2><div class="avasms-fields">';
        foreach ($fields as $f) {
            $this->field_row($f[0], $f[1], (string)$f[2], $f[3], $f[4]);
        }
        echo '</div></div>';
        return (string)ob_get_clean();
    }

    public function page_otp(): void
    {
        $this->shell('avasms-otp', 'تنظیمات OTP', [$this, 'render_otp_inner']);
    }


    public function page_auto(): void
    {
        $this->shell('avasms-auto', 'پیام‌های خودکار', [$this, 'render_auto_inner']);
    }


    public function page_woo(): void
    {
        $this->shell('avasms-woo', 'پیامک‌های ووکامرس', [$this, 'render_woo_inner']);
    }


    public function page_campaign_builder(): void
    {
        $this->shell('avasms-campaign-builder', 'ساخت کمپین', function () {
            $nonce = wp_create_nonce('avasms_campaign');
            $roles = wp_roles()->roles;
            $woo_enabled = !empty(Options::woo()['enabled']) && class_exists('WooCommerce');

            echo '<div class="avasms-panel">';
            echo '<h2>ساخت کمپین پیامکی</h2>';
            echo '<p class="avasms-help">کمپین را ذخیره کنید، سپس از بخش «لیست کمپین‌ها» آن را ارسال کنید (یا زمان‌بندی کنید).</p>';

            echo '<input type="hidden" id="avasms_campaign_nonce" value="' . esc_attr($nonce) . '">';

            echo '<table class="form-table"><tbody>';

            echo '<tr><th><label for="avasms_c_title">نام کمپین</label></th><td>';
            echo '<input type="text" class="regular-text" id="avasms_c_title" placeholder="مثلاً: تخفیف زمستانه">';
            echo '</td></tr>';

            echo '<tr><th><label for="avasms_c_message">متن پیام</label></th><td>';
            echo '<textarea class="large-text" rows="5" id="avasms_c_message" placeholder="متن پیامک..."></textarea>';
            echo '</td></tr>';

            echo '<tr><th><label for="avasms_c_type">نوع مخاطب</label></th><td>';
            echo '<select id="avasms_c_type">';
            echo '<option value="all_users">تمام کاربران</option>';
            echo '<option value="roles">نقش‌های کاربری</option>';
            echo '<option value="manual_list">لیست دستی شماره‌ها</option>';
            echo '<option value="excel">آپلود CSV (mobile,name)</option>';
            echo '<option value="birthday">تبریک تولد (meta)</option>';
            if ($woo_enabled) {
                echo '<option value="woo_all">تمام مشتریان ووکامرس</option>';
                echo '<option value="woo_recent_30">مشتریان ۳۰ روز اخیر</option>';
                echo '<option value="woo_top_buyers">بیشترین خریداران</option>';
            }
            echo '</select>';
            echo '</td></tr>';

            // roles panel
            echo '<tr class="avasms-c-panel" data-panel="roles" style="display:none;"><th>نقش‌ها</th><td>';
            echo '<select id="avasms_c_roles" multiple style="min-width:260px; height:120px;">';
            foreach ($roles as $key => $r) {
                echo '<option value="' . esc_attr($key) . '">' . esc_html($r['name']) . '</option>';
            }
            echo '</select>';
            echo '<p class="description">برای انتخاب چند نقش، کلید Ctrl را نگه دارید.</p>';
            echo '</td></tr>';

            // manual list panel
            echo '<tr class="avasms-c-panel" data-panel="manual_list" style="display:none;"><th>لیست شماره‌ها</th><td>';
            echo '<textarea class="large-text" rows="6" id="avasms_c_numbers" placeholder="هر خط یک شماره..."></textarea>';
            echo '<p class="description">فرمت مجاز: 09xxxxxxxxx یا +98xxxxxxxxxx</p>';
            echo '</td></tr>';

            // csv panel
            echo '<tr class="avasms-c-panel" data-panel="excel" style="display:none;"><th>آپلود CSV</th><td>';
            echo '<input type="file" id="avasms_c_csv" accept=".csv,text/csv,text/plain">';
            echo '<button type="button" class="button" id="avasms_c_upload_csv">آپلود و پیش‌نمایش</button>';
            echo '<div id="avasms_csv_preview" style="margin-top:10px;"></div>';
            echo '</td></tr>';

            // birthday panel
            echo '<tr class="avasms-c-panel" data-panel="birthday" style="display:none;"><th>تبریک تولد</th><td>';
            $copt = Options::campaigns();
            echo '<p class="description">meta key تاریخ تولد: <code>' . esc_html((string)($copt['birthday_meta_key'] ?? 'birthdate')) . '</code> (فرمت YYYY-MM-DD)</p>';
            echo '</td></tr>';

            echo '<tr><th><label for="avasms_c_schedule">زمان‌بندی</label></th><td>';
            echo '<input type="datetime-local" id="avasms_c_schedule">';
            echo '<p class="description">اگر خالی باشد، کمپین به صورت دستی ارسال می‌شود.</p>';
            echo '</td></tr>';

            echo '</tbody></table>';

            echo '<p>';
            echo '<button type="button" class="button button-primary" id="avasms_c_save">ذخیره کمپین</button> ';
            echo '<span id="avasms_c_save_result" class="avasms-status"></span>';
            echo '</p>';

            echo '</div>';
        });
    }

    public function page_campaigns_list(): void
    {
        $this->shell('avasms-campaigns', 'لیست کمپین‌ها', [$this, 'render_campaigns_list_inner']);
    }


    public function page_logs(): void
    {
        $this->shell('avasms-logs', 'لاگ‌ها و گزارشات', [$this, 'render_logs_inner']);
    }


    public function ajax_clear_logs(): void
    {
        if (!current_user_can(self::CAP)) wp_send_json_error(['message' => 'دسترسی غیرمجاز'], 403);
        check_ajax_referer('avasms_logs', 'nonce');

        $days = isset($_POST['days']) ? (int)$_POST['days'] : 30;
        $days = max(1, min(3650, $days));

        $deleted = \Avayemarketing\AvaSMS\Logging\Logger::clear_older_than_days($days);

        wp_send_json_success(['message' => 'پاکسازی انجام شد. تعداد حذف: ' . $deleted]);
    }
public function page_single(): void
{
    $this->shell('avasms-single', 'ارسال تکی پیامک', function () {
        echo '<div class="avasms-panel"><h2>ارسال تکی پیامک</h2><p class="avasms-help">یک کاربر را انتخاب کنید یا شماره را دستی وارد کنید، سپس پیام را ارسال کنید. ارسال از طریق درگاه فعال انجام می‌شود.</p>';

        echo '<div class="avasms-inline avasms-single-wrap">';
        echo '<label style="min-width:140px">انتخاب کاربر</label>';
        echo '<select id="avasms_single_user" class="regular-text">';
        echo '<option value="">— انتخاب —</option>';
        $users = get_users(['number' => 50, 'orderby' => 'registered', 'order' => 'DESC']);
        foreach ($users as $u) {
            $phone = get_user_meta($u->ID, 'billing_phone', true);
            $label = esc_html($u->display_name . ' (' . $u->user_email . ')');
            echo '<option value="'.esc_attr($u->ID).'" data-phone="'.esc_attr((string)$phone).'">'.$label.'</option>';
        }
        echo '</select>';
        echo '</div>';

        echo '<div class="avasms-inline avasms-single-wrap">';
        echo '<label style="min-width:140px">شماره موبایل</label>';
        echo '<input id="avasms_single_phone" type="text" class="regular-text" placeholder="09120000000">';
        echo '</div>';

        echo '<div class="avasms-inline avasms-single-wrap" style="align-items:flex-start">';
        echo '<label style="min-width:140px; padding-top:6px">متن پیام</label>';
        echo '<textarea id="avasms_single_message" class="large-text" rows="4" placeholder="متن پیام را وارد کنید..."></textarea>';
        echo '</div>';

        echo '<div class="avasms-inline avasms-single-wrap">';
        echo '<button class="button button-primary" id="avasms_single_send" data-nonce="'.esc_attr(wp_create_nonce('avasms_send_single')).'">ارسال پیامک</button>';
        echo '<span class="avasms-status"><span class="avasms-dot"></span><span id="avasms_single_result">—</span></span>';
        echo '</div>';

        echo '</div>';
    });
}

public function ajax_test_sms(): void
    {
        if (!current_user_can(self::CAP)) wp_send_json_error(['message' => 'دسترسی غیرمجاز'], 403);
        check_ajax_referer('avasms_test_sms', 'nonce');

        $phone = isset($_POST['phone']) ? sanitize_text_field(wp_unslash($_POST['phone'])) : '';
        $phone = preg_replace('/\D+/', '', $phone);
        if ($phone === '') wp_send_json_error(['message' => 'شماره موبایل نامعتبر است.']);

        $msg = 'پیام تست AvaSMS - ' . date_i18n('Y-m-d H:i');

        $ok = apply_filters('avasms_send_sms', false, $phone, $msg);

        Logger::log($ok ? 'info' : 'error', 'sms-test', 'ارسال پیام تست.', [
            'to' => $phone,
            'ok' => (bool)$ok,
        ]);

        if ($ok) wp_send_json_success(['message' => 'ارسال تست موفق بود.']);
        wp_send_json_error(['message' => 'ارسال تست ناموفق بود (درایور پیامکی هنوز متصل نشده است).']);
    }
public function ajax_send_single(): void
{
    if (!current_user_can(self::CAP)) wp_send_json_error(['message' => 'دسترسی غیرمجاز'], 403);
    check_ajax_referer('avasms_send_single', 'nonce');

    $phone = isset($_POST['phone']) ? sanitize_text_field(wp_unslash($_POST['phone'])) : '';
    $phone = preg_replace('/\D+/', '', $phone);

    $message = isset($_POST['message']) ? sanitize_textarea_field(wp_unslash($_POST['message'])) : '';

    if ($phone === '') wp_send_json_error(['message' => 'شماره موبایل نامعتبر است.']);
    if ($message === '') wp_send_json_error(['message' => 'متن پیام نمی‌تواند خالی باشد.']);

    // Send using active gateway via filter (core will be wired to drivers in SMS module)
    $ok = apply_filters('avasms_send_sms', false, $phone, $message);

    Logger::log($ok ? 'info' : 'error', 'sms-single', 'ارسال تکی پیامک.', [
        'to' => $phone,
        'ok' => (bool)$ok,
    ]);

    if ($ok) wp_send_json_success(['message' => 'ارسال با موفقیت انجام شد.']);
    wp_send_json_error(['message' => 'ارسال ناموفق بود (اتصال درگاه یا درایور را بررسی کنید).']);
}


}
