<?php
namespace Avayemarketing\AvaSMS\Auth;

use Avayemarketing\AvaSMS\Logging\Logger;
use Avayemarketing\AvaSMS\SMS\SmsModule;
use Avayemarketing\AvaSMS\SMS\Utils;
use Avayemarketing\AvaSMS\User\BirthdateField;
use Avayemarketing\AvaSMS\Settings\Options;

if (!defined('ABSPATH')) exit;

/**
 * Frontend SMS OTP auth endpoints used by Elementor widget.
 * This module does NOT remove/override existing features.
 */
final class AuthModule
{
    private const OTP_TTL = 5 * MINUTE_IN_SECONDS; // 5 minutes
    private const OTP_MAX_ATTEMPTS = 5;

    public function register_hooks(): void
    {
        add_action('wp_ajax_avasms_send_otp', [$this, 'ajax_send_otp']);
        add_action('wp_ajax_nopriv_avasms_send_otp', [$this, 'ajax_send_otp']);

        add_action('wp_ajax_avasms_verify_otp', [$this, 'ajax_verify_otp']);
        add_action('wp_ajax_nopriv_avasms_verify_otp', [$this, 'ajax_verify_otp']);
    }

    private function nonce_action(): string
    {
        return 'avasms_auth';
    }

    private function otp_key(string $mobile, string $context): string
    {
        return 'avasms_otp_' . md5($context . '|' . $mobile);
    }

    private function rate_key(string $mobile): string
    {
        return 'avasms_otp_rate_' . md5($mobile);
    }

    public function ajax_send_otp(): void
    {
        if (!check_ajax_referer($this->nonce_action(), 'nonce', false)) {
            wp_send_json_error(['message' => 'درخواست نامعتبر است.'], 403);
        }

        // Optional Turnstile captcha
        $this->maybe_verify_turnstile();

        $context = isset($_POST['context']) ? sanitize_key((string) wp_unslash($_POST['context'])) : 'login';
        if (!in_array($context, ['login', 'register'], true)) $context = 'login';

        // Login can be either by mobile or by email (if enabled)
        $identifier = isset($_POST['identifier']) ? (string) wp_unslash($_POST['identifier']) : '';
        $raw_mobile = isset($_POST['mobile']) ? (string) wp_unslash($_POST['mobile']) : '';
        if ($identifier === '') $identifier = $raw_mobile;

        $mobile = '';
        $user_id = 0;

        if ($context === 'login' && $this->allow_email_login() && is_email($identifier)) {
            $email = sanitize_email($identifier);
            $user = get_user_by('email', $email);
            if (!$user instanceof \WP_User) {
                wp_send_json_error(['message' => 'کاربری با این ایمیل پیدا نشد.'], 404);
            }
            $user_id = (int)$user->ID;
            $mobile = $this->get_user_mobile($user_id);
            if ($mobile === '') {
                wp_send_json_error(['message' => 'برای این ایمیل، شماره موبایل ثبت نشده است.'], 400);
            }
        } else {
            $mobile = Utils::normalize_iran_mobile($identifier);
            if ($mobile === '') {
                wp_send_json_error(['message' => 'شماره موبایل معتبر نیست.'], 400);
            }
        }

        // Basic rate limit: 1 OTP per 60s per mobile
        $rate_key = $this->rate_key($mobile);
        $last = (int) get_transient($rate_key);
        $now = time();
        if ($last && ($now - $last) < 60) {
            wp_send_json_error(['message' => 'لطفاً کمی صبر کنید و دوباره تلاش کنید.'], 429);
        }
        set_transient($rate_key, $now, 60);

        if ($context === 'login') {
            if ($user_id <= 0) {
                $user_id = $this->find_user_by_mobile($mobile);
            }
            if ($user_id <= 0) {
                wp_send_json_error(['message' => 'کاربری با این شماره موبایل پیدا نشد.'], 404);
            }
        } else {
            // register
            $user_id = $this->find_user_by_mobile($mobile);
            if ($user_id > 0) {
                wp_send_json_error(['message' => 'این شماره موبایل قبلاً ثبت شده است.'], 409);
            }
        }

        $otp_opt = Options::otp();
        $len = (int)($otp_opt['length'] ?? 6);
        $len = max(4, min(8, $len));
        $min = (int) pow(10, $len - 1);
        $max = (int) pow(10, $len) - 1;
        $code = (string) wp_rand($min, $max);

        $ttl = (int)($otp_opt['expiry_seconds'] ?? self::OTP_TTL);
        $ttl = max(30, min(900, $ttl));

        $payload = [
            'hash' => wp_hash_password($code),
            'expires' => $now + $ttl,
            'attempts' => 0,
        ];
        set_transient($this->otp_key($mobile, $context), $payload, $ttl);

        $template = isset($_POST['template']) ? sanitize_text_field(wp_unslash($_POST['template'])) : 'کد تایید شما: {code}';
        if ($template === '') $template = 'کد تایید شما: {code}';
        $message = str_replace('{code}', $code, $template);

        $send = (new SmsModule())->send($mobile, $message);

        if (empty($send['success'])) {
            Logger::log('error', 'auth', 'ارسال کد OTP ناموفق بود.', ['mobile'=>$mobile, 'provider'=>$send['provider'] ?? null, 'error_code'=>$send['error_code'] ?? null]);
            wp_send_json_error(['message' => $send['message'] ?? 'ارسال پیامک ناموفق بود.'], 500);
        }

        Logger::log('info', 'auth', 'کد OTP ارسال شد.', ['mobile'=>$mobile, 'context'=>$context]);
        wp_send_json_success(['message' => 'کد تایید ارسال شد.']);
    }

    public function ajax_verify_otp(): void
    {
        if (!check_ajax_referer($this->nonce_action(), 'nonce', false)) {
            wp_send_json_error(['message' => 'درخواست نامعتبر است.'], 403);
        }

        // Optional Turnstile captcha
        $this->maybe_verify_turnstile();

        $context = isset($_POST['context']) ? sanitize_key((string) wp_unslash($_POST['context'])) : 'login';
        if (!in_array($context, ['login', 'register'], true)) $context = 'login';

        $identifier = isset($_POST['identifier']) ? (string) wp_unslash($_POST['identifier']) : '';
        $raw_mobile = isset($_POST['mobile']) ? (string) wp_unslash($_POST['mobile']) : '';
        if ($identifier === '') $identifier = $raw_mobile;

        $mobile = '';
        $user_id = 0;

        if ($context === 'login' && $this->allow_email_login() && is_email($identifier)) {
            $email = sanitize_email($identifier);
            $user = get_user_by('email', $email);
            if (!$user instanceof \WP_User) {
                wp_send_json_error(['message' => 'کاربری با این ایمیل پیدا نشد.'], 404);
            }
            $user_id = (int)$user->ID;
            $mobile = $this->get_user_mobile($user_id);
            if ($mobile === '') {
                wp_send_json_error(['message' => 'برای این ایمیل، شماره موبایل ثبت نشده است.'], 400);
            }
        } else {
            $mobile = Utils::normalize_iran_mobile($identifier);
            if ($mobile === '') {
                wp_send_json_error(['message' => 'شماره موبایل معتبر نیست.'], 400);
            }
        }

        $otp_opt = Options::otp();
        $len = (int)($otp_opt['length'] ?? 6);
        $len = max(4, min(8, $len));

        $code = isset($_POST['code']) ? preg_replace('/\D+/', '', (string) wp_unslash($_POST['code'])) : '';
        if (!is_string($code) || strlen($code) !== $len) {
            wp_send_json_error(['message' => 'کد تایید معتبر نیست.'], 400);
        }

        $key = $this->otp_key($mobile, $context);
        $payload = get_transient($key);

        if (!is_array($payload) || empty($payload['hash']) || empty($payload['expires'])) {
            wp_send_json_error(['message' => 'کد منقضی شده است. دوباره ارسال کنید.'], 410);
        }

        if ((int)($payload['attempts'] ?? 0) >= self::OTP_MAX_ATTEMPTS) {
            delete_transient($key);
            wp_send_json_error(['message' => 'تعداد تلاش بیش از حد مجاز است. دوباره ارسال کنید.'], 429);
        }

        if (time() > (int)$payload['expires']) {
            delete_transient($key);
            wp_send_json_error(['message' => 'کد منقضی شده است. دوباره ارسال کنید.'], 410);
        }

        $payload['attempts'] = (int)($payload['attempts'] ?? 0) + 1;
        set_transient($key, $payload, max(60, (int)$payload['expires'] - time()));

        if (!wp_check_password($code, (string)$payload['hash'])) {
            wp_send_json_error(['message' => 'کد اشتباه است.'], 401);
        }

        // success
        delete_transient($key);

        if ($context === 'login') {
            if ($user_id <= 0) {
                $user_id = $this->find_user_by_mobile($mobile);
            }
            if ($user_id <= 0) {
                wp_send_json_error(['message' => 'کاربر پیدا نشد.'], 404);
            }
        } else {
            $user_id = $this->create_user_from_request($mobile);
            if ($user_id <= 0) {
                wp_send_json_error(['message' => 'ثبت‌نام انجام نشد.'], 500);
            }
        }

        wp_set_current_user($user_id);
        wp_set_auth_cookie($user_id, true);

        Logger::log('info', 'auth', 'ورود/ثبت‌نام موفق.', ['user_id'=>$user_id,'mobile'=>$mobile,'context'=>$context]);

        $redirect = isset($_POST['redirect']) ? esc_url_raw((string) wp_unslash($_POST['redirect'])) : '';
        if ($redirect === '') $redirect = home_url('/');

        wp_send_json_success([
            'message' => 'با موفقیت وارد شدید.',
            'redirect' => $redirect,
        ]);
    }

    private function find_user_by_mobile(string $mobile): int
    {
        // Prefer explicit meta keys
        $q = new \WP_User_Query([
            'meta_query' => [
                'relation' => 'OR',
                [
                    'key' => 'avasms_phone',
                    'value' => $mobile,
                    'compare' => '=',
                ],
                [
                    'key' => 'billing_phone',
                    'value' => $mobile,
                    'compare' => '=',
                ],
            ],
            'number' => 1,
            'fields' => 'ID',
        ]);
        $ids = $q->get_results();
        if (is_array($ids) && !empty($ids[0])) return (int)$ids[0];

        // fallback: username equals mobile (some sites use mobile as login)
        $user = get_user_by('login', $mobile);
        if ($user instanceof \WP_User) return (int)$user->ID;

        return 0;
    }

    private function allow_email_login(): bool
    {
        $opt = Options::otp();
        return !empty($opt['allow_login_with_email']);
    }

    private function get_user_mobile(int $user_id): string
    {
        $m = (string) get_user_meta($user_id, 'avasms_phone', true);
        if ($m === '') $m = (string) get_user_meta($user_id, 'billing_phone', true);
        $m = Utils::normalize_iran_mobile($m);
        return $m;
    }

    private function maybe_verify_turnstile(): void
    {
        $opt = Options::otp();
        if (empty($opt['turnstile_enabled'])) return;
        $secret = (string)($opt['turnstile_secret_key'] ?? '');
        if ($secret === '') return;

        $token = isset($_POST['turnstile_token']) ? sanitize_text_field((string) wp_unslash($_POST['turnstile_token'])) : '';
        if ($token === '') {
            wp_send_json_error(['message' => 'لطفاً کپچا را تکمیل کنید.'], 400);
        }

        $resp = wp_remote_post('https://challenges.cloudflare.com/turnstile/v0/siteverify', [
            'timeout' => 10,
            'body' => [
                'secret' => $secret,
                'response' => $token,
                'remoteip' => isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field((string)$_SERVER['REMOTE_ADDR']) : '',
            ],
        ]);

        if (is_wp_error($resp)) {
            wp_send_json_error(['message' => 'خطا در بررسی کپچا. دوباره تلاش کنید.'], 500);
        }

        $body = wp_remote_retrieve_body($resp);
        $json = json_decode((string)$body, true);
        if (!is_array($json) || empty($json['success'])) {
            wp_send_json_error(['message' => 'کپچا معتبر نیست. دوباره تلاش کنید.'], 400);
        }
    }

    private function create_user_from_request(string $mobile): int
    {
        $first_name = isset($_POST['first_name']) ? sanitize_text_field(wp_unslash($_POST['first_name'])) : '';
        $last_name  = isset($_POST['last_name']) ? sanitize_text_field(wp_unslash($_POST['last_name'])) : '';
        $email      = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
        $birthdate  = isset($_POST['birthdate']) ? sanitize_text_field(wp_unslash($_POST['birthdate'])) : '';

        if ($email && email_exists($email)) {
            // avoid collision; keep empty if already exists
            $email = '';
        }

        $username = $mobile;
        if (username_exists($username)) {
            $username = 'user_' . $mobile . '_' . wp_rand(10, 99);
        }

        $password = wp_generate_password(12, true, true);

        $user_id = wp_insert_user([
            'user_login' => $username,
            'user_pass'  => $password,
            'user_email' => $email,
            'first_name' => $first_name,
            'last_name'  => $last_name,
            'role'       => class_exists('WooCommerce') ? 'customer' : get_option('default_role', 'subscriber'),
        ]);

        if (is_wp_error($user_id)) {
            Logger::log('error', 'auth', 'خطا در ایجاد کاربر.', ['error'=>$user_id->get_error_message()]);
            return 0;
        }

        update_user_meta($user_id, 'avasms_phone', $mobile);
        update_user_meta($user_id, 'billing_phone', $mobile);

        if ($birthdate === '' || preg_match('/^\d{4}-\d{2}-\d{2}$/', $birthdate)) {
            update_user_meta($user_id, BirthdateField::META_KEY, $birthdate);
        }

        $optin = !empty($_POST['marketing_optin']) ? 1 : 0;
        update_user_meta($user_id, 'avasms_marketing_optin', $optin);

        return (int)$user_id;
    }
}
