<?php
namespace Avayemarketing\AvaSMS\Campaigns;

use Avayemarketing\AvaSMS\Logging\Logger;
use Avayemarketing\AvaSMS\Settings\Options;
use Avayemarketing\AvaSMS\SMS\SmsModule;
use Avayemarketing\AvaSMS\SMS\Utils;

if (!defined('ABSPATH')) exit;

final class CampaignsModule
{
    public const TABLE_CAMPAIGNS = 'avasms_campaigns';
    public const TABLE_QUEUE     = 'avasms_campaign_queue';

    public const CRON_HOOK_RUN   = 'avasms_campaign_process_queue';

    public function register_hooks(): void
    {
        // AJAX
        add_action('wp_ajax_avasms_save_campaign', [$this, 'ajax_save_campaign']);
        add_action('wp_ajax_avasms_delete_campaign', [$this, 'ajax_delete_campaign']);
        add_action('wp_ajax_avasms_prepare_campaign', [$this, 'ajax_prepare_campaign']);
        add_action('wp_ajax_avasms_send_campaign_batch', [$this, 'ajax_send_campaign_batch']);
        add_action('wp_ajax_avasms_upload_campaign_csv', [$this, 'ajax_upload_campaign_csv']);

        // Cron
        add_action(self::CRON_HOOK_RUN, [$this, 'cron_process_queue'], 10, 1);
    }

    public function install_tables(): void
    {
        global $wpdb;
        $charset = $wpdb->get_charset_collate();

        $campaigns = self::table_campaigns();
        $queue     = self::table_queue();

        $sql1 = "CREATE TABLE {$campaigns} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            title VARCHAR(255) NOT NULL,
            message TEXT NOT NULL,
            audience_type VARCHAR(50) NOT NULL,
            audience_meta LONGTEXT NULL,
            scheduled_at DATETIME NULL,
            status VARCHAR(20) NOT NULL DEFAULT 'draft',
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME NULL,
            PRIMARY KEY (id),
            KEY status (status),
            KEY scheduled_at (scheduled_at)
        ) {$charset};";

        $sql2 = "CREATE TABLE {$queue} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            campaign_id BIGINT(20) UNSIGNED NOT NULL,
            recipient VARCHAR(30) NOT NULL,
            name VARCHAR(190) NULL,
            status VARCHAR(20) NOT NULL DEFAULT 'pending',
            result LONGTEXT NULL,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            sent_at DATETIME NULL,
            PRIMARY KEY (id),
            KEY campaign_id (campaign_id),
            KEY status (status)
        ) {$charset};";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql1);
        dbDelta($sql2);
    }

    public static function table_campaigns(): string
    {
        global $wpdb;
        return $wpdb->prefix . self::TABLE_CAMPAIGNS;
    }

    public static function table_queue(): string
    {
        global $wpdb;
        return $wpdb->prefix . self::TABLE_QUEUE;
    }

    // ---------------- AJAX: Save Campaign ----------------
    public function ajax_save_campaign(): void
    {
        if (!current_user_can('manage_options')) wp_send_json_error(['message' => 'دسترسی غیرمجاز'], 403);
        check_ajax_referer('avasms_campaign', 'nonce');

        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        $title = isset($_POST['title']) ? sanitize_text_field(wp_unslash($_POST['title'])) : '';
        $message = isset($_POST['message']) ? sanitize_textarea_field(wp_unslash($_POST['message'])) : '';
        $audience_type = isset($_POST['audience_type']) ? sanitize_key(wp_unslash($_POST['audience_type'])) : 'manual';
        $audience_meta_raw = isset($_POST['audience_meta']) ? wp_unslash($_POST['audience_meta']) : '';

        $scheduled_at = isset($_POST['scheduled_at']) ? sanitize_text_field(wp_unslash($_POST['scheduled_at'])) : '';
        $scheduled_at = trim($scheduled_at);

        $audience_meta = [];
        if (is_string($audience_meta_raw) && $audience_meta_raw !== '') {
            $decoded = json_decode($audience_meta_raw, true);
            if (is_array($decoded)) $audience_meta = $decoded;
        }

        if ($title === '' || $message === '') {
            wp_send_json_error(['message' => 'نام کمپین و متن پیام الزامی است.']);
        }

        $allowed_types = ['all_users','roles','manual_list','woo_all','woo_recent_30','woo_top_buyers','birthday','excel'];
        if (!in_array($audience_type, $allowed_types, true)) $audience_type = 'manual_list';

        $status = $scheduled_at !== '' ? 'scheduled' : 'draft';

        global $wpdb;
        $table = self::table_campaigns();

        $data = [
            'title' => $title,
            'message' => $message,
            'audience_type' => $audience_type,
            'audience_meta' => !empty($audience_meta) ? wp_json_encode($audience_meta, JSON_UNESCAPED_UNICODE) : null,
            'scheduled_at' => $scheduled_at !== '' ? gmdate('Y-m-d H:i:s', strtotime($scheduled_at)) : null,
            'status' => $status,
            'updated_at' => current_time('mysql'),
        ];

        if ($id > 0) {
            $wpdb->update($table, $data, ['id' => $id]);
        } else {
            $data['created_at'] = current_time('mysql');
            $wpdb->insert($table, $data);
            $id = (int)$wpdb->insert_id;
        }

        if ($status === 'scheduled' && $id > 0) {
            $this->schedule_campaign($id, $data['scheduled_at']);
        }

        wp_send_json_success(['message' => 'کمپین ذخیره شد.', 'id' => $id, 'status' => $status]);
    }

    // ---------------- AJAX: Delete Campaign ----------------
    public function ajax_delete_campaign(): void
    {
        if (!current_user_can('manage_options')) wp_send_json_error(['message' => 'دسترسی غیرمجاز'], 403);
        check_ajax_referer('avasms_campaign', 'nonce');

        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) wp_send_json_error(['message' => 'شناسه نامعتبر است.']);

        global $wpdb;
        $wpdb->delete(self::table_campaigns(), ['id' => $id], ['%d']);
        $wpdb->delete(self::table_queue(), ['campaign_id' => $id], ['%d']);

        wp_clear_scheduled_hook(self::CRON_HOOK_RUN, [$id]);

        wp_send_json_success(['message' => 'کمپین حذف شد.']);
    }

    // ---------------- AJAX: Prepare Queue ----------------
    public function ajax_prepare_campaign(): void
    {
        if (!current_user_can('manage_options')) wp_send_json_error(['message' => 'دسترسی غیرمجاز'], 403);
        check_ajax_referer('avasms_campaign', 'nonce');

        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) wp_send_json_error(['message' => 'شناسه کمپین نامعتبر است.']);

        $count = $this->build_queue_for_campaign($id, true);

        wp_send_json_success(['message' => 'لیست مخاطبین آماده شد.', 'queued' => $count]);
    }

    // ---------------- AJAX: Send Batch ----------------
    public function ajax_send_campaign_batch(): void
    {
        if (!current_user_can('manage_options')) wp_send_json_error(['message' => 'دسترسی غیرمجاز'], 403);
        check_ajax_referer('avasms_campaign', 'nonce');

        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) wp_send_json_error(['message' => 'شناسه کمپین نامعتبر است.']);

        $batch = (int)Options::campaigns()['batch_size'];
        $sent = $this->send_batch($id, $batch);

        $stats = $this->queue_stats($id);

        wp_send_json_success([
            'message' => 'ارسال دسته‌ای انجام شد.',
            'sent_now' => $sent,
            'stats' => $stats,
        ]);
    }

    // ---------------- AJAX: Upload CSV ----------------
    public function ajax_upload_campaign_csv(): void
    {
        if (!current_user_can('manage_options')) wp_send_json_error(['message' => 'دسترسی غیرمجاز'], 403);
        check_ajax_referer('avasms_campaign', 'nonce');

        if (empty($_FILES['file'])) wp_send_json_error(['message' => 'فایل ارسال نشده است.']);

        require_once ABSPATH . 'wp-admin/includes/file.php';

        $uploaded = wp_handle_upload($_FILES['file'], ['test_form' => false, 'mimes' => [
            'csv'  => 'text/csv',
            'txt'  => 'text/plain',
        ]]);

        if (isset($uploaded['error'])) {
            wp_send_json_error(['message' => (string)$uploaded['error']]);
        }

        $path = (string)($uploaded['file'] ?? '');
        if ($path === '' || !is_readable($path)) {
            wp_send_json_error(['message' => 'فایل قابل خواندن نیست.']);
        }

        $rows = $this->parse_csv($path);
        if (empty($rows)) {
            wp_send_json_error(['message' => 'هیچ رکوردی در فایل یافت نشد. (ستون mobile الزامی است)']);
        }

        $preview = array_slice($rows, 0, 10);

        wp_send_json_success([
            'message' => 'فایل دریافت شد.',
            'rows' => $rows,
            'preview' => $preview,
            'note' => 'فرمت XLSX در این نسخه پشتیبانی نمی‌شود. لطفاً CSV آپلود کنید.',
        ]);
    }

    private function parse_csv(string $path): array
    {
        $out = [];
        $fh = fopen($path, 'r');
        if (!$fh) return [];
        $header = null;

        while (($row = fgetcsv($fh)) !== false) {
            if ($header === null) {
                $header = array_map(function($h){
                    $h = strtolower(trim((string)$h));
                    return $h;
                }, $row);
                continue;
            }
            $assoc = [];
            foreach ($header as $i => $k) {
                $assoc[$k] = isset($row[$i]) ? trim((string)$row[$i]) : '';
            }

            $mobile = Utils::normalize_iran_mobile((string)($assoc['mobile'] ?? ($assoc['phone'] ?? '')));
            if ($mobile === '') continue;

            $out[] = [
                'mobile' => $mobile,
                'name'   => sanitize_text_field((string)($assoc['name'] ?? '')),
            ];
        }
        fclose($fh);
        return $out;
    }

    // ---------------- Queue Builder ----------------
    private function build_queue_for_campaign(int $campaign_id, bool $wipe_previous = false): int
    {
        $campaign = $this->get_campaign($campaign_id);
        if (!$campaign) return 0;

        if ($wipe_previous) {
            global $wpdb;
            $wpdb->delete(self::table_queue(), ['campaign_id' => $campaign_id], ['%d']);
        }

        $type = (string)$campaign['audience_type'];
        $meta = [];
        if (!empty($campaign['audience_meta'])) {
            $decoded = json_decode((string)$campaign['audience_meta'], true);
            if (is_array($decoded)) $meta = $decoded;
        }

        $recipients = [];

        // Woo features toggle
        $woo_enabled = !empty(Options::woo()['enabled']) && class_exists('WooCommerce');

        switch ($type) {
            case 'all_users':
                $recipients = $this->audience_all_users();
                break;

            case 'roles':
                $roles = isset($meta['roles']) && is_array($meta['roles']) ? array_map('sanitize_key', $meta['roles']) : [];
                $recipients = $this->audience_roles($roles);
                break;

            case 'manual_list':
                $list = (string)($meta['numbers'] ?? '');
                $recipients = $this->audience_manual_list($list);
                break;

            case 'excel':
                $rows = isset($meta['rows']) && is_array($meta['rows']) ? $meta['rows'] : [];
                $recipients = $this->audience_excel_rows($rows);
                break;

            case 'birthday':
                $recipients = $this->audience_birthdays();
                break;

            case 'woo_all':
                if ($woo_enabled) $recipients = $this->audience_woo_all_customers();
                break;

            case 'woo_recent_30':
                if ($woo_enabled) $recipients = $this->audience_woo_recent_customers(30);
                break;

            case 'woo_top_buyers':
                if ($woo_enabled) $recipients = $this->audience_woo_top_buyers();
                break;
        }

        // de-dupe
        $seen = [];
        $clean = [];
        foreach ($recipients as $r) {
            $m = Utils::normalize_iran_mobile((string)($r['mobile'] ?? ''));
            if ($m === '' || isset($seen[$m])) continue;
            $seen[$m] = true;
            $clean[] = [
                'mobile' => $m,
                'name'   => sanitize_text_field((string)($r['name'] ?? '')),
            ];
        }

        if (empty($clean)) return 0;

        global $wpdb;
        $table = self::table_queue();

        foreach ($clean as $r) {
            $wpdb->insert($table, [
                'campaign_id' => $campaign_id,
                'recipient' => $r['mobile'],
                'name' => $r['name'] !== '' ? $r['name'] : null,
                'status' => 'pending',
                'result' => null,
                'created_at' => current_time('mysql'),
            ], ['%d','%s','%s','%s','%s','%s']);
        }

        return count($clean);
    }

    private function get_campaign(int $id): ?array
    {
        global $wpdb;
        $row = $wpdb->get_row($wpdb->prepare("SELECT * FROM " . self::table_campaigns() . " WHERE id=%d", $id), ARRAY_A);
        return is_array($row) ? $row : null;
    }

    public function list_campaigns(int $limit = 100): array
    {
        global $wpdb;
        $limit = max(1, min(500, $limit));
        $rows = $wpdb->get_results("SELECT * FROM " . self::table_campaigns() . " ORDER BY id DESC LIMIT {$limit}", ARRAY_A);
        return is_array($rows) ? $rows : [];
    }

    private function queue_stats(int $campaign_id): array
    {
        global $wpdb;
        $table = self::table_queue();

        $total = (int)$wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$table} WHERE campaign_id=%d", $campaign_id));
        $pending = (int)$wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$table} WHERE campaign_id=%d AND status='pending'", $campaign_id));
        $sent = (int)$wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$table} WHERE campaign_id=%d AND status='sent'", $campaign_id));
        $failed = (int)$wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$table} WHERE campaign_id=%d AND status='failed'", $campaign_id));

        return compact('total','pending','sent','failed');
    }

    // ---------------- Sending ----------------
    private function send_batch(int $campaign_id, int $batch_size): int
    {
        $batch_size = max(1, min(200, $batch_size));

        $campaign = $this->get_campaign($campaign_id);
        if (!$campaign) return 0;

        global $wpdb;
        $table = self::table_queue();

        $items = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$table} WHERE campaign_id=%d AND status='pending' ORDER BY id ASC LIMIT %d",
            $campaign_id,
            $batch_size
        ), ARRAY_A);

        if (empty($items)) {
            // mark campaign as sent if any sent or no pending
            $stats = $this->queue_stats($campaign_id);
            if ($stats['total'] > 0 && $stats['pending'] === 0) {
                $wpdb->update(self::table_campaigns(), ['status' => ($stats['failed'] > 0 ? 'failed' : 'sent')], ['id' => $campaign_id]);
            }
            return 0;
        }

        $sms = new SmsModule();
        $sent_now = 0;

        foreach ($items as $it) {
            $to = (string)$it['recipient'];
            $msg = (string)$campaign['message'];

            $res = $sms->send($to, $msg);

            $new_status = !empty($res['success']) ? 'sent' : 'failed';
            $sent_at = !empty($res['success']) ? current_time('mysql') : null;

            $wpdb->update($table, [
                'status' => $new_status,
                'result' => wp_json_encode($res, JSON_UNESCAPED_UNICODE),
                'sent_at' => $sent_at,
            ], ['id' => (int)$it['id']], ['%s','%s','%s'], ['%d']);

            Logger::log(!empty($res['success']) ? 'info' : 'error', 'campaign', 'ارسال کمپین', [
                'campaign_id' => $campaign_id,
                'to' => $to,
                'success' => !empty($res['success']),
                'provider' => $res['provider'] ?? null,
                'error_code' => $res['error_code'] ?? null,
            ]);

            if (!empty($res['success'])) $sent_now++;
        }

        // update campaign status
        $stats = $this->queue_stats($campaign_id);
        if ($stats['pending'] === 0) {
            $wpdb->update(self::table_campaigns(), ['status' => ($stats['failed'] > 0 ? 'failed' : 'sent')], ['id' => $campaign_id]);
        } else {
            $wpdb->update(self::table_campaigns(), ['status' => 'scheduled'], ['id' => $campaign_id]);
        }

        return $sent_now;
    }

    private function schedule_campaign(int $campaign_id, ?string $scheduled_at_gmt): void
    {
        if (!$scheduled_at_gmt) return;

        $ts = strtotime($scheduled_at_gmt . ' UTC');
        if (!$ts) return;

        // schedule a single event to start processing
        if (!wp_next_scheduled(self::CRON_HOOK_RUN, [$campaign_id])) {
            wp_schedule_single_event($ts, self::CRON_HOOK_RUN, [$campaign_id]);
        }
    }

    public function cron_process_queue(int $campaign_id): void
    {
        // Build queue once if empty
        $stats = $this->queue_stats($campaign_id);
        if ($stats['total'] === 0) {
            $this->build_queue_for_campaign($campaign_id, true);
        }

        $batch = (int)Options::campaigns()['batch_size'];
        $this->send_batch($campaign_id, $batch);

        $stats = $this->queue_stats($campaign_id);
        if ($stats['pending'] > 0) {
            // Continue in one minute
            wp_schedule_single_event(time() + 60, self::CRON_HOOK_RUN, [$campaign_id]);
        }
    }

    // ---------------- Audiences ----------------
    private function audience_all_users(): array
    {
        $users = get_users(['fields' => ['ID'], 'number' => 5000]);
        $out = [];
        foreach ($users as $u) {
            $uid = is_object($u) ? (int)$u->ID : (int)$u;
            $mobile = $this->get_user_mobile($uid);
            if ($mobile !== '') $out[] = ['mobile' => $mobile, 'name' => $this->get_user_name($uid)];
        }
        return $out;
    }

    private function audience_roles(array $roles): array
    {
        if (empty($roles)) return [];
        $users = get_users(['role__in' => $roles, 'fields' => ['ID'], 'number' => 5000]);
        $out = [];
        foreach ($users as $u) {
            $uid = is_object($u) ? (int)$u->ID : (int)$u;
            $mobile = $this->get_user_mobile($uid);
            if ($mobile !== '') $out[] = ['mobile' => $mobile, 'name' => $this->get_user_name($uid)];
        }
        return $out;
    }

    private function audience_manual_list(string $textarea): array
    {
        $lines = preg_split('/\r\n|\r|\n/', (string)$textarea) ?: [];
        $out = [];
        foreach ($lines as $l) {
            $m = Utils::normalize_iran_mobile((string)$l);
            if ($m !== '') $out[] = ['mobile' => $m, 'name' => ''];
        }
        return $out;
    }

    private function audience_excel_rows(array $rows): array
    {
        $out = [];
        foreach ($rows as $r) {
            if (!is_array($r)) continue;
            $m = Utils::normalize_iran_mobile((string)($r['mobile'] ?? ''));
            if ($m === '') continue;
            $out[] = ['mobile' => $m, 'name' => (string)($r['name'] ?? '')];
        }
        return $out;
    }

    private function audience_birthdays(): array
    {
        $opt = Options::campaigns();
        $key = sanitize_key((string)($opt['birthday_meta_key'] ?? 'birthdate'));

        $today = current_time('Y-m-d');
        $parts = explode('-', $today);
        $mmdd = (count($parts) === 3) ? ($parts[1] . '-' . $parts[2]) : '';

        $users = get_users(['fields' => ['ID'], 'number' => 5000]);
        $out = [];

        foreach ($users as $u) {
            $uid = is_object($u) ? (int)$u->ID : (int)$u;
            $bd = (string)get_user_meta($uid, $key, true);
            if ($bd === '') continue;

            // expect YYYY-MM-DD
            if (strlen($bd) >= 10 && substr($bd, 5, 5) === $mmdd) {
                $mobile = $this->get_user_mobile($uid);
                if ($mobile !== '') $out[] = ['mobile' => $mobile, 'name' => $this->get_user_name($uid)];
            }
        }

        return $out;
    }

    private function audience_woo_all_customers(): array
    {
        if (!function_exists('wc_get_orders')) return [];
        $orders = wc_get_orders([
            'limit' => 2000,
            'status' => ['completed','processing','on-hold'],
            'orderby' => 'date',
            'order' => 'DESC',
            'return' => 'objects',
        ]);

        $out = [];
        foreach ($orders as $o) {
            $phone = Utils::normalize_iran_mobile((string)$o->get_billing_phone());
            if ($phone === '') continue;
            $out[] = ['mobile' => $phone, 'name' => (string)$o->get_billing_first_name()];
        }
        return $out;
    }

    private function audience_woo_recent_customers(int $days): array
    {
        if (!function_exists('wc_get_orders')) return [];
        $after = (new \DateTimeImmutable('now', wp_timezone()))->modify('-' . max(1,$days) . ' days');

        $orders = wc_get_orders([
            'limit' => 2000,
            'status' => ['completed'],
            'date_created' => '>' . $after->format('Y-m-d H:i:s'),
            'return' => 'objects',
        ]);

        $out = [];
        foreach ($orders as $o) {
            $phone = Utils::normalize_iran_mobile((string)$o->get_billing_phone());
            if ($phone === '') continue;
            $out[] = ['mobile' => $phone, 'name' => (string)$o->get_billing_first_name()];
        }
        return $out;
    }

    private function audience_woo_top_buyers(): array
    {
        if (!function_exists('wc_get_orders')) return [];
        $opt = Options::campaigns();
        $limit = max(5, min(500, (int)($opt['top_buyers_limit'] ?? 50)));

        $orders = wc_get_orders([
            'limit' => 5000,
            'status' => ['completed'],
            'return' => 'objects',
        ]);

        $agg = [];
        foreach ($orders as $o) {
            $phone = Utils::normalize_iran_mobile((string)$o->get_billing_phone());
            if ($phone === '') continue;
            $total = (float)$o->get_total();

            if (!isset($agg[$phone])) {
                $agg[$phone] = [
                    'mobile' => $phone,
                    'name' => (string)$o->get_billing_first_name(),
                    'total' => 0.0,
                    'count' => 0,
                ];
            }
            $agg[$phone]['total'] += $total;
            $agg[$phone]['count'] += 1;
        }

        uasort($agg, function($a,$b){
            return ($b['total'] <=> $a['total']);
        });

        $out = [];
        foreach (array_slice($agg, 0, $limit) as $r) {
            $out[] = ['mobile' => $r['mobile'], 'name' => (string)$r['name']];
        }
        return $out;
    }

    private function get_user_mobile(int $user_id): string
    {
        $keys = apply_filters('avasms_user_mobile_meta_keys', [
            'billing_phone',
            'phone',
            'mobile',
            'user_phone',
            'avasms_mobile',
        ]);

        if (!is_array($keys)) $keys = ['billing_phone'];

        foreach ($keys as $k) {
            $v = (string)get_user_meta($user_id, sanitize_key((string)$k), true);
            $m = Utils::normalize_iran_mobile($v);
            if ($m !== '') return $m;
        }

        return '';
    }

    private function get_user_name(int $user_id): string
    {
        $u = get_userdata($user_id);
        if (!$u) return '';
        $name = trim((string)$u->display_name);
        return $name;
    }
}
