/*!
 * ue-ztext.js v0.0.3
 * This library is customized version of ztext.js v0.0.3(https://bennettfeely.com/ztext), Please check before updaing.
 * Customization: Responsive depth on different screen sizes by Adarsh Pawar
 * fork of https://bennettfeely.com/ztext
 * Licensed MIT | (c) 2020 Bennett Feely
 */

if (
  CSS.supports("-moz-transform-style", "preserve-3d") ||
  CSS.supports("-ms-transform-style", "preserve-3d") ||
  CSS.supports("-webkit-transform-style", "preserve-3d") ||
  CSS.supports("transform-style", "preserve-3d")
) {
  // Default values with responsive depth
  z_default = {
    depth: {
      mobile: "0.5rem", // For screens <= 768px
      tablet: "0.8rem", // For screens <= 1024px
      desktop: "1rem", // For screens > 1024px
    },
    direction: "both",
    event: "none",
    eventRotation: "30deg",
    eventDirection: "default",
    fade: false,
    layers: 10,
    perspective: "500px",
    z: true,
  };

  // Store instances for cleanup
  let zInstances = new Map();

  // Debounce function
  function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
      const later = () => {
        clearTimeout(timeout);
        func(...args);
      };
      clearTimeout(timeout);
      timeout = setTimeout(later, wait);
    };
  }

  // Get responsive depth based on window width
  function getResponsiveDepth(depthOptions) {
    const width = window.innerWidth;
    if (width <= 768) return depthOptions.mobile;
    if (width <= 1024) return depthOptions.tablet;
    return depthOptions.desktop;
  }

  // Handle window resize
  const handleResize = debounce(() => {
    zInstances.forEach((instance, element) => {
      // Get new depth based on current window size
      const newDepth = getResponsiveDepth(instance.options.depth);

      // Only reinitialize if depth changed
      if (newDepth !== instance.currentDepth) {
        // Create temporary container for new layers
        const tempContainer = document.createElement("div");
        tempContainer.style.opacity = "0";
        tempContainer.style.position = "absolute";
        tempContainer.style.top = "0";
        tempContainer.style.left = "0";
        tempContainer.style.width = "100%";
        tempContainer.style.height = "100%";

        // Update options with new depth
        const updatedOptions = {
          ...instance.options,
          depth: newDepth,
        };

        // Create new layers in temporary container
        zDrawInContainer(element, tempContainer, updatedOptions);

        // Add temporary container to element
        element.appendChild(tempContainer);

        // Get old and new z-text elements
        const oldZText = element.querySelector(":scope > .z-text");
        const newZText = tempContainer.querySelector(".z-text");

        if (oldZText && newZText) {
          // Move new z-text out of temp container
          element.appendChild(newZText);

          // Smooth transition
          requestAnimationFrame(() => {
            newZText.style.opacity = "1";
            if (oldZText) {
              oldZText.style.transition = "opacity 0.2s";
              oldZText.style.opacity = "0";

              // Remove old content after transition
              setTimeout(() => {
                if (oldZText.parentNode === element) {
                  element.removeChild(oldZText);
                }
              }, 200);
            }
          });
        }

        // Remove temporary container
        element.removeChild(tempContainer);

        // Update instance tracking
        zInstances.set(element, {
          options: instance.options,
          currentDepth: newDepth,
        });
      }
    });
  }, 150);

  window.addEventListener("resize", handleResize);

  // Get all elements with the [data-z] attribute
  var zs = document.querySelectorAll("[data-z]");
  zs.forEach((z) => {
    // Make uniform option keys with responsive depth
    const dataOptions = {
      text: z.dataset.zText || z.innerHTML,
      depth: {
        mobile: z.dataset.zDepthMobile || z_default.depth.mobile,
        tablet: z.dataset.zDepthTablet || z_default.depth.tablet,
        desktop: z.dataset.zDepthDesktop || z_default.depth.desktop,
      },
      direction: z.dataset.zDirection || z_default.direction,
      event: z.dataset.zEvent || z_default.event,
      eventRotation: z.dataset.zEventrotation || z_default.eventRotation,
      eventDirection: z.dataset.zEventdirection || z_default.eventDirection,
      fade: z.dataset.zFade || z_default.fade,
      layers: parseFloat(z.dataset.zLayers) || z_default.layers,
      perspective: z.dataset.zPerspective || z_default.perspective,
      zEngaged: z.dataset.z || z_default.z,
    };

    const initialDepth = getResponsiveDepth(dataOptions.depth);
    const drawOptions = {
      ...dataOptions,
      depth: initialDepth,
    };

    zDraw(z, drawOptions);

    // Store instance
    zInstances.set(z, {
      options: dataOptions,
      currentDepth: initialDepth,
    });
  });

  // JS constructor
  function Ztextify(selector, options) {
    const elements = document.querySelectorAll(selector);

    elements.forEach((element) => {
      // Merge default and custom options
      const mergedOptions = {
        ...z_default,
        ...options,
        depth: {
          ...z_default.depth,
          ...(options?.depth || {}),
        },
      };

      // Get initial depth based on current window size
      const initialDepth = getResponsiveDepth(mergedOptions.depth);

      // Initialize with current depth
      const drawOptions = {
        ...mergedOptions,
        depth: initialDepth,
      };

      zDraw(element, drawOptions);

      // Store instance for future updates
      zInstances.set(element, {
        options: mergedOptions,
        currentDepth: initialDepth,
      });
    });
  }

  function zDrawInContainer(z, container, options) {
    var z_engaged = options.zEngaged || z_default.z;

    if (z_engaged !== "false") {
      var depth = options.depth || z_default.depth.desktop;
      var depth_unit =
        typeof depth === "object"
          ? z_default.depth.desktop.match(/[a-z]+/)[0]
          : depth.match(/[a-z]+/)[0];
      var depth_numeral =
        typeof depth === "object"
          ? parseFloat(z_default.depth.desktop.replace(depth_unit, ""))
          : parseFloat(depth.replace(depth_unit, ""));

      var direction = options.direction || z_default.direction;
      var event = options.event || z_default.event;
      var event_rotation = options.eventRotation || z_default.eventRotation;
      var event_rotation_unit = event_rotation.match(/[a-z]+/)[0];
      var event_rotation_numeral = parseFloat(
        event_rotation.replace(event_rotation_unit, "")
      );
      var event_direction = options.eventDirection || z_default.eventDirection;
      var fade = options.fade || z_default.fade;
      var layers = options.layers || z_default.layers;
      var perspective = options.perspective || z_default.perspective;

      // Get text content
      var text = options.text || z.innerHTML;

      if (container === z) {
        // Only clear if we're drawing in the original container
        z.innerHTML = "";
      }

      // Set container styles
      container.style.display = "inline-block";
      container.style.position = "relative";
      container.style.webkitPerspective = perspective;
      container.style.perspective = perspective;

      // Create wrapper span
      var zText = document.createElement("span");
      zText.setAttribute("class", "z-text");
      zText.style.display = "inline-block";
      zText.style.webkitTransformStyle = "preserve-3d";
      zText.style.transformStyle = "preserve-3d";

      // Create layers container
      var zLayers = document.createElement("span");
      zLayers.setAttribute("class", "z-layers");
      zLayers.style.display = "inline-block";
      zLayers.style.webkitTransformStyle = "preserve-3d";
      zLayers.style.transformStyle = "preserve-3d";

      zText.append(zLayers);

      for (var i = 0; i < layers; i++) {
        let pct = i / layers;

        // Create a layer
        var zLayer = document.createElement("span");
        zLayer.setAttribute("class", "z-layer");
        zLayer.innerHTML = text;
        zLayer.style.display = "inline-block";

        // Shift the layer on the z axis
        if (direction === "backwards") {
          var zTranslation = -pct * depth_numeral;
        }
        if (direction === "both") {
          var zTranslation = -(pct * depth_numeral) + depth_numeral / 2;
        }
        if (direction === "forwards") {
          var zTranslation = -(pct * depth_numeral) + depth_numeral;
        }

        var transform = "translateZ(" + zTranslation + depth_unit + ")";
        zLayer.style.webkitTransform = transform;
        zLayer.style.transform = transform;

        // Manipulate duplicate layers
        if (i >= 1) {
          zLayer.style.position = "absolute";
          zLayer.style.top = 0;
          zLayer.style.left = 0;
          zLayer.setAttribute("aria-hidden", "true");
          zLayer.style.pointerEvents = "none";
          zLayer.style.mozUserSelect = "none";
          zLayer.style.msUserSelect = "none";
          zLayer.style.webkitUserSelect = "none";
          zLayer.style.userSelect = "none";

          if (fade === true || fade === "true") {
            zLayer.style.opacity = (1 - pct) / 2;
          }
        }

        zLayers.append(zLayer);
      }

      container.append(zText);

      function tilt(x_pct, y_pct) {
        if (event_direction == "reverse") {
          var event_direction_adj = -1;
        } else {
          var event_direction_adj = 1;
        }

        var x_tilt = x_pct * event_rotation_numeral * event_direction_adj;
        var y_tilt = -y_pct * event_rotation_numeral * event_direction_adj;

        var x_clamped = Math.min(Math.max(x_tilt, -1), 1);
        var y_clamped = Math.min(Math.max(y_tilt, -1), 1);

        var transform =
          "rotateX(" +
          y_tilt +
          event_rotation_unit +
          ") rotateY(" +
          x_tilt +
          event_rotation_unit +
          ")";
        zLayers.style.webkitTransform = transform;
        zLayers.style.transform = transform;
      }

      // Event handlers
      if (event === "pointer") {
        window.addEventListener(
          "mousemove",
          (e) => {
            var x_pct = (e.clientX / window.innerWidth - 0.5) * 2;
            var y_pct = (e.clientY / window.innerHeight - 0.5) * 2;
            tilt(x_pct, y_pct);
          },
          false
        );

        window.addEventListener(
          "touchmove",
          (e) => {
            var x_pct = (e.touches[0].clientX / window.innerWidth - 0.5) * 2;
            var y_pct = (e.touches[0].clientY / window.innerHeight - 0.5) * 2;
            tilt(x_pct, y_pct);
          },
          false
        );
      }

      if (event == "scroll") {
        function zScroll() {
          var bounds = z.getBoundingClientRect();
          var center_x = bounds.left + bounds.width / 2 - window.innerWidth / 2;
          var center_y =
            bounds.top + bounds.height / 2 - window.innerHeight / 2;
          var x_pct = (center_x / window.innerWidth) * -2;
          var y_pct = (center_y / window.innerHeight) * -2;
          tilt(x_pct, y_pct);
        }

        zScroll();
        window.addEventListener("scroll", zScroll, false);
      }

      if (event == "scrollY") {
        function zScrollY() {
          var bounds = z.getBoundingClientRect();
          var center_y =
            bounds.top + bounds.height / 2 - window.innerHeight / 2;
          var y_pct = (center_y / window.innerHeight) * -2;
          tilt(0, y_pct);
        }

        zScrollY();
        window.addEventListener("scroll", zScrollY, false);
      }

      if (event == "scrollX") {
        function zScrollX() {
          var bounds = z.getBoundingClientRect();
          var center_x = bounds.left + bounds.width / 2 - window.innerWidth / 2;
          var x_pct = (center_x / window.innerWidth) * -2;
          tilt(x_pct, 0);
        }

        zScrollX();
        window.addEventListener("scroll", zScrollX, false);
      }

      return zText;
    }
  }

  function zDraw(z, options) {
    return zDrawInContainer(z, z, options);
  }
} else {
  console.error(
    "ztext is disabled because transform-style: preserve-3d; is unsupported"
  );
}
